'use strict';

var Game = {};

/**
 * The supported languages.
 */
Game.LANGUAGE_NAME = {
	'zh': '中文',
	'en': 'English'
};

Game.workspace = null;

Game.MAX_LEVEL = 10;

/*
返回一个介于最小值和最大值之间的值
 */
Game.clamp = function(min, val, max) 
{
	if(val < min)
	{
		val = min;
	}
	else if(val > max)
	{
		val = max;
	}

	return val;
};

/*
从URL中提取参数
 */
Game.getStringParamFromUrl = function(name, defaultValue) 
{
	var val = location.search.match(new RegExp('[?&]' + name + '=([^&]+)'));
	return val ? decodeURIComponent(val[1].replace(/\+/g, '%20')) : defaultValue;
};

/*
从URL中提取数字参数
 */
Game.getNumberParamFromUrl = function(name, minValue, maxValue) 
{
    var val = Number(Game.getStringParamFromUrl(name, 'NaN'));
    return isNaN(val) ? minValue : Game.clamp(minValue, val, maxValue);
};

/*
从URL获取此用户的语言
 */
Game.getLang = function() {
	var lang = Game.getStringParamFromUrl('lang', 'zh');
	if(Game.LANGUAGE_NAME[lang] === undefined) 
	{
		lang = 'zh';
	}
	return lang;
};

/*
从URL获取此游戏的级别
 */
Game.getLevel = function() 
{
	var level = Game.getNumberParamFromUrl('level', 1, Game.MAX_LEVEL);
	return level;
};

/*
从URL获取此游戏的名称
*/
Game.getGameName = function() 
{
	var name = Game.getStringParamFromUrl('name', 'maze');
	return name;
};


Game.LANG = Game.getLang();
Game.LEVEL = Game.getLevel();
Game.NAME = Game.getGameName();

/*
加载保存在本地存储中的块
 */
Game.loadBlocks = function(defaultXml) 
{
	try 
	{
		var loadOnce = window.sessionStorage.loadOnceBlocks;
	} 
	catch(e) 
	{
		var loadOnce = null;
	}
	if('BlocklyStorage' in window && window.location.hash.length > 1) 
	{
		BlocklyStorage.retrieveXml(window.location.hash.substring(1));
	} 
	else if(loadOnce) 
	{
		delete window.sessionStorage.loadOnceBlocks;
		var xml = Blockly.Xml.textToDom(loadOnce);
		Blockly.Xml.domToWorkspace(xml, Game.workspace);
	} 
	else if(defaultXml) 
	{
		var xml = Blockly.Xml.textToDom(defaultXml);
		Game.workspace.clear();
		Blockly.Xml.domToWorkspace(xml, Game.workspace);
		Game.workspace.clearUndo();
	} 
	else if('BlocklyStorage' in window) 
	{
		window.setTimeout(BlocklyStorage.restoreBlocks, 0);
	}
};

/*
保存块并用其他语言重新加载
 */
Game.changeLanguage = function() 
{
	if(typeof Blockly != 'undefined' && window.sessionStorage) 
	{
		if(Game.NAME != 'tank')
		{
			var xml = Blockly.Xml.workspaceToDom(Game.workspace);
		}
		else 
		{
			var xml = Blockly.Xml.workspaceToDom(DDBlockly.workspace);
		}
		
		var text = Blockly.Xml.domToText(xml);
		window.sessionStorage.loadOnceBlocks = text;
	}

	var languageMenu = document.getElementById('languageMenu');
	var newLang = encodeURIComponent(languageMenu.options[languageMenu.selectedIndex].value);
	var search = window.location.search;
	if(search.length <= 1) 
	{
		search = '?lang=' + newLang + '&level=' + Game.LEVEL + '&name=' + Game.NAME;
	} 
	else if(search.match(/[?&]lang=[^&]*/)) 
	{
		search = search.replace(/([?&]lang=)[^&]*/, '$1' + newLang);
	} 
	else 
	{
		search = search.replace(/\?/, '?lang=' + newLang + '&');
	}

	window.location = window.location.protocol + '//' + window.location.host + window.location.pathname + search;
};

/*
将函数绑定到按钮的单击事件
 */
Game.bindClick = function(el, func) 
{
	if(typeof el == 'string') 
	{
		el = document.querySelector(el);
	}
	el.addEventListener('click', func, true);
	el.addEventListener('touchend', func, true);
};

/*
创建关卡链接按钮
 */
Game.displayLevelLink = function() 
{
	var levelLink = document.getElementById('levelLink');
	var a = null,button = null;
	var cur = null;

	var wateraction = function() 
	{
		var cur = $(this);
		var dest = cur.position().left;
		var t = 0.4;
		dest -= 50 * (Game.LEVEL - 0)-15;
		TweenMax.to($('.select'), t, { x: dest, ease: Back.easeOut });
	//		动态获得关卡数字
		$('.select').html(cur.html());
	};

	//  动态生成关卡按钮
	for(var i = 1; i <= Game.MAX_LEVEL; ++i) 
	{
		a = document.createElement('a');
		a.innerHTML = i;
		a.href = '?lang=' + Game.LANG + '&level=' + i + '&name=' + Game.NAME;
		a.addEventListener('mouseover', wateraction);
		a.classList.add('levelbtn');
		levelLink.appendChild(a);

		if(i === Game.LEVEL)
		{
			cur = $(a);
		}
	}
	//	鼠标离开dots时,select返回选中关卡
	var dest = cur.position().left;
	var t = 0.4;
	dest -= 50 * (Game.LEVEL - 0)-15;
	$('.dots').mouseleave(function()
	{
		TweenMax.to($('.select'), t, { x: dest, ease: Back.easeOut });
		$('.select').html(cur.html());
	})
	//	选中关卡后,水滴停留在对应关卡
	$('.select').css('left', cur.position().left + 16).html(Game.LEVEL);
};

/*
绑定按钮点击事件
*/
Game.btnEvent = function() 
{
	/*
  // 运行和重置按钮
	var btnRun = document.getElementById('playBtn');
	//var btnReset = document.getElementById('resetBtn');
	var btnEvent = function() 
	{
		if(btnRun.textContent == MSG['play'])
		{
			btnRun.textContent == MSG['reset'];
		}
		else
		{
			btnRun.textContent == MSG['play'];
		}
		btnRun.classList.toggle('active');
		//btnReset.classList.toggle('active');
	};
	Game.bindClick(btnRun, btnEvent);
	//Game.bindClick(btnReset, btnEvent);
*/
  // 切换代码按钮
	var codebtnEvent = function()
	{
		var dialogCode = document.getElementById('dialogCode');
		var dialogP = document.querySelector('#dialogCode pre');
		var layer = document.getElementsByClassName('layer')[0];
		var code = Blockly.JavaScript.workspaceToCode(Game.workspace);
		code = code.replace(/(,\s*)?'block_id_[^']+'\)/g, ')');
		if(typeof prettyPrintOne == 'function') {
			code = prettyPrintOne(code, 'js');
		}	
		dialogP.innerHTML = code;
		dialogCode.style.display = 'block';
		layer.style.display = 'block';
	};
	Game.bindClick('.showcode', codebtnEvent);

	// 对话框按钮
	var btnConfirm = document.getElementsByClassName('dialog-btn');
	
	Game.bindClick(btnConfirm[0], function() 
	{
		Game.hideDialog('dialogCode');
	});
	Game.bindClick(btnConfirm[1], function() 
	{
		Game.hideDialog('dialogTip')
	});
	Game.bindClick(btnConfirm[2], function() 
	{
		Game.hideDialog('dialogWin')
	});
	Game.bindClick(btnConfirm[3], Game.nextLevel);
};

/*
 * 加载Prettify CSS和JavaScript.
 */
Game.importPrettify = function() 
{
  var link = document.createElement('link');
  link.setAttribute('rel', 'stylesheet');
  link.setAttribute('href', 'prettify/prettify.css');
  document.head.appendChild(link);
  var script = document.createElement('script');
  script.setAttribute('src', 'prettify/prettify.js');
  document.head.appendChild(script);
};

Game.resize = function() 
{
	var sUserAgent = navigator.userAgent.toLowerCase();
	if (/ipad|iphone|midp|rv:1.2.3.4|ucweb|android|windows ce|windows mobile/.test(sUserAgent)) 
	{//移动端 
		document.body.style.zoom =0.95;
		return 1;
	} 
	else 
	{//pc端
		return 0;
	}
}

/**
 * 初始化Blockly.
 */
Game.init = function() 
{	
	Game.resize();
	Game.initLanguage();
	Game.displayLevelLink();
	Game.btnEvent();
	Game.loadBlocks('');

	if('BlocklyStorage' in window) 
	{
		BlocklyStorage.backupOnUnload(Game.workspace);
	}
  	Blockly.Blocks && (Blockly.Blocks.ONE_BASED_INDEXING = false);
  	Blockly.JavaScript && (Blockly.JavaScript.ONE_BASED_INDEXING = false);

	// 延迟加载语法高亮显示
	window.setTimeout(Game.importPrettify, 1);
};

/**
 * 初始化语言.
 */
Game.initLanguage = function() 
{
	document.head.parentElement.setAttribute('lang', Game.LANG);
	var languages = [];
	for(var lang in Game.LANGUAGE_NAME) 
	{
		languages.push([Game.LANGUAGE_NAME[lang], lang]);
	}

	var languageMenu = document.getElementById('languageMenu');	
	languageMenu.options.length = 0;
	for(var i = 0; i < languages.length; i++) 
	{
		var tuple = languages[i];
		var lang = tuple[tuple.length - 1];
		var option = new Option(tuple[0], lang);
		if(lang == Game.LANG) 
		{
			option.selected = true;
		}
		languageMenu.options.add(option);
	}
	languageMenu.addEventListener('change', Game.changeLanguage, true);

	// 注入语言字符串
	document.getElementById('playBtn').textContent = MSG['play'];
	//document.getElementById('resetBtn').textContent = MSG['reset'];
	document.getElementsByClassName('showcode')[0].textContent = MSG['showcode'];
	//document.getElementById('html_index').textContent = MSG['index'];
	//document.querySelector('#innertop_name h3').textContent = MSG[Game.NAME];
	document.querySelector('#dialogCode .dialog-h').textContent = MSG['code'];
	document.querySelector('#dialogCode #dialogCodeBtn').textContent = MSG['sure'];
	document.querySelector('#dialogTip .dialog-btn').textContent = MSG['sure'];
	document.querySelector('#dialogWin .dialog-h').textContent = MSG['success'];
	document.querySelector('#dialogWin .dialog-btn-left').textContent = MSG['replay'];
	document.querySelector('#dialogWin .dialog-btn-right').textContent = MSG['nextLevel'];
	document.querySelector('#popover button').textContent = MSG['reduceDiff'];
};

/**
* 加载游戏JavaScript.
*/
Game.importGameScript = function() 
{
	var script = '';
	if (Game.NAME == 'maze') 
	{
		script = ['<script type="text/javascript" src="js/maze.js"></script>\n',
		'<script type="text/javascript" src="js/maze.source.js"></script>\n',
		'<script type="text/javascript" src="js/maze.draw.js"></script>\n',
		'<script type="text/javascript" src="js/maze.core.js"></script>\n'].join('');
	}
	else if (Game.NAME == 'painting') 
	{
		script = '<script type="text/javascript" src="js/painting.js"></script>';
	}
	else 
	{
		script = ['<script src="./js/lib/jquery.min.js"></script>\n',
		'<script src="./js/plugins/json-2.3.min.js"></script>\n',
		'<script src="./js/crafty-dev.js"></script>',
		'<script src="./js/soundmanager.js"></script>',
		'<script src="./js/class.js"></script>',
		'<script src="./js/classes/settings.js"></script>',
		'<script src="./js/classes/sound.js"></script>',
		'<script src="./js/classes/app.js"></script>',
		// '<script src="./js/classes/overlay.js"></script>',
		'<script src="./js/classes/program.js"></script>',
		'<script src="./js/classes/blocklyprogram.js"></script>',
		'<script src="./js/classes/level.js"></script>',
		'<script src="./js/classes/map_object.js"></script>',
		'<script src="./js/classes/map.js"></script>',
		'<script src="./js/classes/dir.js"></script>',
		'<script src="./js/classes/pos.js"></script>',
		'<script src="./js/classes/tank.js"></script>',
		'<script src="./js/classes/mine.js"></script>',
		'<script src="./js/classes/base.js"></script>',
		'<script src="./js/classes/tree.js"></script>',
		'<script src="./js/classes/rock.js"></script>',
		'<script src="./js/classes/wall.js"></script>',
		'<script src="./js/classes/bullet.js"></script>',
		'<script src="./js/classes/beam_tower.js"></script>',
		'<script src="./js/classes/beam.js"></script>',
		'<script src="./js/classes/explosion.js"></script>',
		'<script src="./blocks/tank.js"></script>',
		'<script src="./js/planeWar.js"></script>',
	  	'<script src="./js/planeWar.app.js"></script>'].join('');
	}
	document.write(script);
};

/**
* ----------------------------------------------
* Common functions for game below.
* ----------------------------------------------
*/

/*
初始化工作区
 */
Game.initWorkspace = function(maxBlocks) 
{
	//var toolboxText = document.getElementById('toolbox').outerHTML;
	//toolboxText = toolboxText.replace(/{(\w+)}/g, function(m, p1) { return MSG[p1]; });
	//var toolboxXml = Blockly.Xml.textToDom(toolboxText);

	var toolbox = document.getElementById('toolbox');
	if (maxBlocks == void 0) 
	{
		maxBlocks = Infinity;
	}

	Game.workspace = Blockly.inject('workspce_block', {
		grid: 
		{
			spacing: 25,
			length: 3,
			colour: '#ccc',
			snap: true
		},
		maxBlocks: maxBlocks,
		media: 'media/',
		toolbox:toolbox,// toolboxXml,
		trashcan: true,
		zoom:
		{
			controls: true,
			startScale: 1.0,
			maxScale: 3,
			minScale: 0.3,
            wheel: false
		}
	});
};

/*
初始化工具箱
 */
Game.initToolbox = function(game) 
{
	var toolbox = document.getElementById('toolbox');
	var block = null;
	var blocks = [];

	blocks = game.blocks[Game.LEVEL - 1];

	for(var index in blocks) 
	{
		block = document.createElement('block');
		block.setAttribute('type', blocks[index]);
		toolbox.appendChild(block);
	}
};

/*
高亮显示块（或清除高亮显示）
 */
Game.highlight = function(id) 
{
	if (id) {
    var m = id.match(/^block_id_([^']+)$/);
    if (m) {
      id = m[1];
    }
  }
  Game.workspace.highlightBlock(id);
};

/*
预加载图像
*/
Game.loadImages = function(src, onComplete) 
{
	var num = 0
	Game.imgs = [];
	for(var i in src) {
		Game.imgs[i] = new Image();
		Game.imgs[i].src = src[i];
		Game.imgs[i].onload = function() {
			num ++;
			if(num >= src.length) {
				onComplete();
			}
		};
	}
};

/*
下一关
*/
Game.nextLevel = function() 
{
    if (Game.LEVEL < Game.MAX_LEVEL) {
        window.location = window.location.protocol + '//' +
        window.location.host + window.location.pathname +'?lang=' + Game.LANG + '&level=' + (Game.LEVEL + 1) + '&name=' + Game.NAME;
    } 
    else 
    {
    	alert('Exceed max level!!!!');
    }
};

/*
显示弹窗
*/
Game.showDialog = function(id) 
{
	var dialog = document.getElementById(id);
	var layer = document.getElementsByClassName('layer')[0];
	dialog.style.display = 'block';
	layer.style.display = 'block';
};

/*
关闭弹窗
*/
Game
Game.hideDialog = function(id) 
{
	var dialog = document.getElementById(id);
	var layer = document.getElementsByClassName('layer')[0];
	dialog.style.display = 'none';
	layer.style.display = 'none';
}

// 重写现有方法时发出警告
if(Array.prototype.equals)
    console.warn("覆盖现有数组、原型等。可能的原因：新API定义了方法，存在框架冲突，或者代码中包含了双重内容");
// attach the .equals method to Array's prototype to call it on any array
Array.prototype.equals = function (array) {
    // if the other array is a falsy value, return
    if (!array)
        return false;

    // compare lengths - can save a lot of time
    if (this.length != array.length)
        return false;

    for (var i = 0, l = this.length; i < l; i++) {
        // Check if we have nested arrays
        if (this[i] instanceof Array && array[i] instanceof Array) {
            // recurse into the nested arrays
            if (!this[i].equals(array[i]))
                return false;
        }
        else if (this[i] != array[i]) {
            // Warning - two different object instances will never be equal: {x:20} != {x:20}
            return false;
        }
    }
    return true;
}
// Hide method from for-in loops
Object.defineProperty(Array.prototype, "equals", {enumerable: false});


document.write('<script src="msg/lang_' + Game.LANG + '.js"></script>\n');
document.write('<script src="blockly/msg/js/' + Game.LANG + '.js"></script>\n');
document.write('<script src="msg/messages_' + Game.LANG + '.js"></script>\n');
document.write('<script src="msg/dialogContent_' + Game.LANG + '.js"></script>\n');
// Load game JavaScript.
Game.importGameScript();

window.addEventListener('load', Game.init);
